import nodes
import unittest
import cStringIO

class Test1(unittest.TestCase):

    def setUp(self):
        self.h = nodes.Hierarchy()
        f = cStringIO.StringIO("""
nodes: n[1-99]
enclosure e1: n[1-4]
enclosure e2: n[5-8]
enclosure e3: n[9-12]
enclosure e4: n[13-16]
rack r1: e[1-2]
rack r2: e[3-4]
oddball o1: n[6-10]
""")
        self.h.parse_file(f)
        f.close()

    def expands_to(self, nodespec, hostlist):
        self.assertEqual(self.h.expand(nodespec), hostlist)
        
    def expands_bad(self, nodespec):
        self.assertRaises(nodes.NodesException,
                          self.h.expand,  nodespec)

    def test_expand(self):
        self.expands_to("",       "")
        self.expands_to("n1",     "n1")
        self.expands_to("n99",    "n99")
        self.expands_to("e1",     "n[1-4]")
        self.expands_to("e1,e1",  "n[1-4]")
        self.expands_to("e[1-2]", "n[1-8]")
        self.expands_to("e[1,3]", "n[1-4,9-12]")
        self.expands_to("e1,n1",  "n[1-4]")
        self.expands_to("e1,n5",  "n[1-5]")
        self.expands_to("n6,e1",  "n[1-4,6]")
        self.expands_to("r1",     "n[1-8]")
        self.expands_to("r1,e3",  "n[1-12]")
        self.expands_to("r1,e4",  "n[1-8,13-16]")
        self.expands_to("o1",     "n[6-10]")
        self.expands_to("o1,e1",  "n[1-4,6-10]")
        self.expands_to("o1,e2",  "n[5-10]")
        self.expands_to("o1,e3",  "n[6-12]")
        
        self.expands_bad("n0")
        self.expands_bad("n100")

    def ufg(self, level, nodespec, up, fill, gather):
        if up is not None:
            self.assertEqual(self.h.up(level, nodespec), up)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.up, level, nodespec)

        if fill is not None:
            self.assertEqual(self.h.fill(level, nodespec), fill)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.fill, level, nodespec)

        if gather is not None:
            self.assertEqual(self.h.gather(level, nodespec), gather)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.gather, level, nodespec)

    def ufg_e(self, nodespec, up, fill, gather):
        self.ufg("enclosure", nodespec, up, fill, gather)
        
    def ufg_r(self, nodespec, up, fill, gather):
        self.ufg("rack", nodespec, up, fill, gather)

    def test_up_fill_gather(self):
        #          NODESPEC     UP          FILL            GATHER
        # -------------------------------------------------------------------
        self.ufg_e("",          "",         "",             "")
        self.ufg_e("e1",        "e1",       "n[1-4]",       "e1")  
        self.ufg_e("n1",        "e1",       "n[1-4]",       "n1")
        self.ufg_e("n2",        "e1",       "n[1-4]",       "n2")
        self.ufg_e("n3",        "e1",       "n[1-4]",       "n3")
        self.ufg_e("n4",        "e1",       "n[1-4]",       "n4")
        self.ufg_e("n5",        "e2",       "n[5-8]",       "n5")
        self.ufg_e("n[1-2]",    "e1",       "n[1-4]",       "n[1-2]")
        self.ufg_e("n[4-5]",    "e[1-2]",   "n[1-8]",       "n[4-5]")
        self.ufg_e("n[1-5,14]", "e[1-2,4]", "n[1-8,13-16]", "e1,n[5,14]")
        self.ufg_e("n[1-8,14]", "e[1-2,4]", "n[1-8,13-16]", "e[1-2],n14")
        self.ufg_e("n[1-9,14]", "e[1-4]",   "n[1-16]",      "e[1-2],n[9,14]")

        self.ufg_e("r1",        "e[1-2]",   "n[1-8]",       "e[1-2]")
        self.ufg_e("r1,e3",     "e[1-3]",   "n[1-12]",      "e[1-3]")
        self.ufg_e("r1,e4",     "e[1-2,4]", "n[1-8,13-16]", "e[1-2,4]")
        self.ufg_e("r1,n16",    "e[1-2,4]", "n[1-8,13-16]", "e[1-2],n16")
        self.ufg_e("r[1-2]",    "e[1-4]",   "n[1-16]",      "e[1-4]")

        self.ufg_e("o1",        "e[2-3]",   "n[5-12]",      "n[6-10]")

        self.ufg_e("e5",        None,       None,           None)
        self.ufg_e("n17",       None,       "n17",          "n17")
        self.ufg_e("n[16-17]",  None,       "n[13-17]",     "n[16-17]")

        self.ufg_r("r1",        "r1",       "n[1-8]",       "r1")
        self.ufg_r("e1",        "r1",       "n[1-8]",       "n[1-4]")
        self.ufg_r("n1",        "r1",       "n[1-8]",       "n1")
        self.ufg_r("e[1-2]",    "r1",       "n[1-8]",       "r1")
        self.ufg_r("e[1-3]",    "r[1-2]",   "n[1-16]",      "n[9-12],r1")
        self.ufg_r("e[1-4]",    "r[1-2]",   "n[1-16]",      "r[1-2]")
        self.ufg_r("r1,e4",     "r[1-2]",   "n[1-16]",      "n[13-16],r1")
        self.ufg_r("r1,n16",    "r[1-2]",   "n[1-16]",      "n16,r1")

        self.ufg_r("o1",        "r[1-2]",   "n[1-16]",      "n[6-10]")

        self.ufg_r("r3",        None,       None,           None)
        self.ufg_r("n17",       None,       "n17",          "n17")
        self.ufg_r("n[16-17]",  None,       "n[9-17]",      "n[16-17]")

if __name__ == '__main__':
    unittest.main()

        
